/************************************************************************************************************\

Module Name:    MosaicModule.c

Description:    .

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

// -----------------------------------------------------------------------------------------------------------
//                                  I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "MosaicModule.h"
#include "VpeModule.h"

// -----------------------------------------------------------------------------------------------------------
//                         N A M E S P A C E ,   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                        S T A T I C   M E M B E R S   I N I T I A L I S A T I O N
// -----------------------------------------------------------------------------------------------------------

static const    MCHAR8      g_szModuleNameBase[]    = "Mosaic";
static          MUINT32     g_uiMosaicModCount      = 0;

// -----------------------------------------------------------------------------------------------------------
//                                                  C O D E
// -----------------------------------------------------------------------------------------------------------

/************************************************************************************************************\

Function:       MosMod_InvalidateBuffers

Description:    .

\************************************************************************************************************/
void MosMod_InvalidateBuffers(MosaicModule* poMosMod)
{
    MUINT i = 0;

    for (i = 0; i < poMosMod->oOutLink.uiBufferCount; i++)
    {
        if (poMosMod->oOutLink.aoBufferInfo[i].pvPrivateData != MNULL)
        {
            *((MBOOL*)(poMosMod->oOutLink.aoBufferInfo[i].pvPrivateData)) = MFALSE;
        }
    }
}

/************************************************************************************************************\

Function:       MosMod_Init

Description:    .

\************************************************************************************************************/
LStatus MosMod_Init(
            MosaicModule*           poMosMod,
            LDevice_Handle          hDevice,
            MUINT                   uiOutBufferCount,
            LBuffer_Attributes*     poOutBufferAttributes,
            LSIZE*                  poOutFrameSize,
            MosMod_SubmitNotifyFn   pfnSubmitNotify,
            MBOOL                   bSyncInputs,
            MBOOL                   bLowLatency)
{
    MsgLog(2, "{...");

    MosMod_Cleanup(poMosMod);

    LStatus eStatus = ((poMosMod != MNULL)
                       && (hDevice != MNULL)
                       && (poOutBufferAttributes != MNULL)
                       && (poOutBufferAttributes->eAttributeType == LBuffer_Type_VIDEO))
                      ? LStatus_OK : LStatus_INVALID_PARAM;

    MUINT i;
    for (i = 0; (i < MOSAIC_DT_COUNT) && LSTATUS_IS_SUCCESS(eStatus); i++)
    {
        LDeviceThread_CreateAttributes oAttributes = {LDeviceThread_AttributeType_CREATE, MTRUE, MTRUE, MFALSE, "Mosaic0"};

        snprintf(oAttributes.szName, sizeof(oAttributes.szName), "msc%d", i);

        eStatus = LDeviceThread_Create(hDevice, &oAttributes.eType, &(poMosMod->ahDevThread[i]));

        if (LSTATUS_IS_FAIL(eStatus))
        {
            MsgLogErr("ERROR! LDeviceThread_Create failed.");
        }
    }

    for (i = 0; (i < MOSAIC_VPE_COUNT) && LSTATUS_IS_SUCCESS(eStatus); i++)
    {
        eStatus = LVPE_Create(
                        hDevice,
                        poMosMod->ahDevThread[MOSAIC_VPE_DT_IDX + i],
                        MNULL,
                        &(poMosMod->ahVpe[i]));
    }

    for (i = 0; (i < MOSAIC_VPC_COUNT) && LSTATUS_IS_SUCCESS(eStatus); i++)
    {
        eStatus = LVPC_Create(
                        hDevice,
                        poMosMod->ahDevThread[MOSAIC_VPC_DT_IDX + i],
                        MNULL,
                        &(poMosMod->ahVpc[i]));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LBlit_Create(hDevice, poMosMod->ahDevThread[MOSAIC_VPE_DT_IDX], MNULL, &(poMosMod->hBlit));
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        snprintf(
                    poMosMod->szModuleName,
                    sizeof(poMosMod->szModuleName),
                    "%s%d",
                    g_szModuleNameBase,
                    g_uiMosaicModCount);

        eStatus = ModLnk_Init(
                        &(poMosMod->oOutLink),
                        hDevice,
                        uiOutBufferCount,
                        poOutBufferAttributes,
                        MFALSE,
                        sizeof(MBOOL),
                        poMosMod->szModuleName);
    }

    LVP_ColorSpace oColorSpace;

    oColorSpace.eFunction    = LVP_TransferFunction_Other;
    oColorSpace.eMatrix      = LVP_TransferMatrix_BT709;
    oColorSpace.poUserMatrix = MNULL;

    MsgLog(2, "DstTransFn= %d", oColorSpace.eFunction);

    for (i = 0; (i < MOSAIC_VPE_COUNT) && LSTATUS_IS_SUCCESS(eStatus); i++)
    {
        eStatus = LVPE_SetTargetParam(
                    poMosMod->ahVpe[i],
                    LVPE_Target_COLOR_SPACE,
                    sizeof(LVP_ColorSpace),
                    &oColorSpace);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPE_SetSourceParam(
                            poMosMod->ahVpe[i],
                            LVPE_Source_COLOR_SPACE,
                            sizeof(LVP_ColorSpace),
                            &oColorSpace);
        }
    }

    for (i = 0; (i < MOSAIC_VPC_COUNT) && LSTATUS_IS_SUCCESS(eStatus); i++)
    {
        eStatus = LVPC_SetTargetParam(
                    poMosMod->ahVpc[i],
                    LVPC_Target_COLOR_SPACE,
                    sizeof(LVP_ColorSpace),
                    &oColorSpace);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPC_SetSourceParam(
                            poMosMod->ahVpc[i],
                            0,
                            LVPC_Source_COLOR_SPACE,
                            sizeof(LVP_ColorSpace),
                            &oColorSpace);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LVPC_SetSourceParam(
                            poMosMod->ahVpc[i],
                            1,
                            LVPC_Source_COLOR_SPACE,
                            sizeof(LVP_ColorSpace),
                            &oColorSpace);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            LVPC_CompositorColorSpace oCompColSpace;

            oCompColSpace.bYCbCrA = LPixelFormat_IsRgb(poOutBufferAttributes->oVideoAttributes.ePixelFormat)
                                    ? MFALSE : MTRUE;
            oCompColSpace.oColorSpace = oColorSpace;

            eStatus = LVPC_SetCompositorParam(
                        poMosMod->ahVpc[i],
                        LVPC_Compositor_COLOR_SPACE,
                        sizeof(LVPC_CompositorColorSpace),
                        &oCompColSpace);
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = LClipRgn_Create(&poMosMod->hFullClipRgn);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        poMosMod->oFullDstRect.bEnable      = MTRUE;
        poMosMod->oFullDstRect.oRect.iLeft  = 0;
        poMosMod->oFullDstRect.oRect.iTop   = 0;

        if ((poOutFrameSize != MNULL)
            && (poOutFrameSize->iWidth <= poOutBufferAttributes->oVideoAttributes.uiWidth)
            && (poOutFrameSize->iHeight <= poOutBufferAttributes->oVideoAttributes.uiHeight))
        {
            poMosMod->oFullDstRect.oRect.iRight  = poOutFrameSize->iWidth;
            poMosMod->oFullDstRect.oRect.iBottom = poOutFrameSize->iHeight;
        }
        else
        {
            poMosMod->oFullDstRect.oRect.iRight  = poOutBufferAttributes->oVideoAttributes.uiWidth;
            poMosMod->oFullDstRect.oRect.iBottom = poOutBufferAttributes->oVideoAttributes.uiHeight;
        }

        eStatus = LClipRgn_SetRect(poMosMod->hFullClipRgn, &poMosMod->oFullDstRect.oRect, LBlit_Bop_S);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        poMosMod->uiOutFrameRateNum = 1;
        poMosMod->uiOutFrameRateDen = 1;
        poMosMod->pfnSubmitNotify   = pfnSubmitNotify;
        poMosMod->bSyncInputs       = bSyncInputs;
        poMosMod->bLowLatency       = bLowLatency;

        for (i = 0; i < MOSAIC_MAX_INPUT; i++)
        {
            poMosMod->aoInLink[i].bSyncBefore = bLowLatency ? MFALSE : MTRUE;
        }

        MosMod_InvalidateBuffers(poMosMod);
    }
    else
    {
        MosMod_Cleanup(poMosMod);
    }

    ++g_uiMosaicModCount;

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       MosMod_Cleanup

Description:    .

\************************************************************************************************************/
void MosMod_Cleanup(MosaicModule* poMosMod)
{
    MsgLog(2, "{...");

    if (poMosMod != MNULL)
    {
        ModLnk_Cleanup(&(poMosMod->oOutLink));

        if (poMosMod->hFullClipRgn != MNULL)
        {
            LClipRgn_Destroy(poMosMod->hFullClipRgn);
            poMosMod->hFullClipRgn = MNULL;
        }

        if (poMosMod->hBlit != MNULL)
        {
            while (LSTATUS_IS_FAIL(LBlit_Destroy(poMosMod->hBlit)))
            {
                MsgLog(0, "LBlit_Destroy failed! Retry...");
                usleep(1000*1000);
            }

            poMosMod->hBlit = MNULL;
        }

        MUINT i;
        for (i = 0; i < MOSAIC_VPC_COUNT; i++)
        {
            if (poMosMod->ahVpc[i] != MNULL)
            {
                while (LSTATUS_IS_FAIL(LVPC_Destroy(poMosMod->ahVpc[i])))
                {
                    MsgLog(0, "LVPC_Destroy failed! Retry...");
                    usleep(1000*1000);
                }
            }

            poMosMod->ahVpc[i] = MNULL;
        }

        for (i = 0; i < MOSAIC_VPE_COUNT; i++)
        {
            if (poMosMod->ahVpe[i] != MNULL)
            {
                while (LSTATUS_IS_FAIL(LVPE_Destroy(poMosMod->ahVpe[i])))
                {
                    MsgLog(0, "LVPE_Destroy failed! Retry...");
                    usleep(1000*1000);
                }
            }

            poMosMod->ahVpe[i] = MNULL;
        }

        for (i = 0; i < MOSAIC_DT_COUNT; i++)
        {
            if (poMosMod->ahDevThread[i] != MNULL)
            {
                LDeviceThread_Destroy(poMosMod->ahDevThread[i]);
            }

            poMosMod->ahDevThread[i] = MNULL;
        }

        for (i = 0; i < MOSAIC_MAX_INPUT; i++)
        {
            if (poMosMod->bInitParam)
            {
                if (poMosMod->aoInData[i].hClipRgn != MNULL)
                {
                    LClipRgn_Destroy(poMosMod->aoInData[i].hClipRgn);
                }
            }

            poMosMod->aoInData[i].hClipRgn                  = MNULL;
            poMosMod->aoInLink[i].poModLnk                  = MNULL;
            poMosMod->aoInParam[i].oSrcFrameSize.iWidth     = 1920;
            poMosMod->aoInParam[i].oSrcFrameSize.iHeight    = 1080;
            poMosMod->aoInParam[i].oSrcRect.bEnable         = MFALSE;
            poMosMod->aoInParam[i].oDstRect.bEnable         = MFALSE;
            poMosMod->aoInParam[i].bUseVpc                  = MFALSE;
            poMosMod->aoInParam[i].uiFrameRateNum           = 60;
            poMosMod->aoInParam[i].uiFrameRateDen           = 1;
            poMosMod->aoInParam[i].uiZOrder                 = 0;
            poMosMod->aoInParam[i].bControlQueueOverflow    = MFALSE;
            poMosMod->aoInParam[i].fAlpha                   = 1.0f;
        }

        poMosMod->bInitParam           = MTRUE;
        poMosMod->bShowFullScreen      = MFALSE;
        poMosMod->uiFullScreenInputIdx = 0;
        poMosMod->pfnSubmitNotify      = MNULL;
    }

    MsgLog(2, "...}");
}

/************************************************************************************************************\

Function:       MosMod_SetParameters

Description:    .

\************************************************************************************************************/
LStatus MosMod_SetParameters(
    MosaicModule*           poMosMod,
    MUINT                   uiInputIndex,
    const LSIZE*            poSrcFrameSize,
    const LRECT32*          poSrcRect,
    const LRECT32*          poDstRect,
    MUINT                   uiFrameRateNum,
    MUINT                   uiFrameRateDen,
    MUINT                   uiZOrder,
    MBOOL                   bControlQueueOverflow,
    MFLOAT32                fAlpha)
{
    LStatus eStatus = LStatus_INVALID_PARAM;

    MsgLog(2, "{...");

    if ((poMosMod != MNULL)
        && (uiInputIndex < MOSAIC_MAX_INPUT)
        && (uiFrameRateNum > 0)
        && (uiFrameRateDen > 0)
        && (poSrcFrameSize != MNULL))
    {
        eStatus = LStatus_OK;

        MosaicInputParam* poParam = poMosMod->aoInParam + uiInputIndex;

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            if (poSrcRect != MNULL)
            {
                poParam->oSrcRect.bEnable   = MTRUE;
                poParam->oSrcRect.oRect     = *poSrcRect;
            }
            else
            {
                poParam->oSrcRect.bEnable   = MFALSE;
            }
        }

        MBOOL bDoScaling = MTRUE;

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            if (poDstRect != MNULL)
            {
                poParam->oDstRect.bEnable = MTRUE;
                poParam->oDstRect.oRect   = *poDstRect;
            }
            else
            {
                poParam->oDstRect = poMosMod->oFullDstRect;
            }

            if ((poSrcRect != MNULL)
                && ((poParam->oDstRect.oRect.iRight - poParam->oDstRect.oRect.iLeft)
                    == (poSrcRect->iRight - poSrcRect->iLeft))
                && ((poParam->oDstRect.oRect.iBottom - poParam->oDstRect.oRect.iTop)
                    == (poSrcRect->iBottom - poSrcRect->iTop)))
            {
                bDoScaling = MFALSE;
            }
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            // Cannot scale and blend at the same time.
            if (bDoScaling)
            {
                if (fAlpha == 1.0f)
                {
                    poParam->bUseVpc = MFALSE;
                }
                else
                {
                    eStatus = LStatus_INVALID_PARAM;
                }
            }
            else
            {
                poParam->bUseVpc = MTRUE;
            }
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            poParam->oSrcFrameSize          = *poSrcFrameSize;
            poParam->uiFrameRateNum         = uiFrameRateNum;
            poParam->uiFrameRateDen         = uiFrameRateDen;
            poParam->uiZOrder               = uiZOrder;
            poParam->bControlQueueOverflow  = bControlQueueOverflow;
            poParam->fAlpha                 = fAlpha;

            MFLOAT32 fFrameRate     = (MFLOAT32)uiFrameRateNum / uiFrameRateDen;
            MFLOAT32 fOutFrameRate  = (MFLOAT32)poMosMod->uiOutFrameRateNum / poMosMod->uiOutFrameRateDen;

            if (fFrameRate > fOutFrameRate)
            {
                poMosMod->uiOutFrameRateNum = uiFrameRateNum;
                poMosMod->uiOutFrameRateDen = uiFrameRateDen;
            }

            MsgLog(2, "Input[%u]: FrameRate= %u/%u, Z= %u, OverflowCtl= %d",
                      uiInputIndex, uiFrameRateNum, uiFrameRateDen, uiZOrder, bControlQueueOverflow);
        }
    }

    MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));

    return eStatus;
}

/************************************************************************************************************\

Function:       MosMod_GetOutputFrameRate

Description:    .

\************************************************************************************************************/
LStatus MosMod_GetOutputFrameRate(
    MosaicModule*   poMosMod,
    MUINT*          puiFrameRateNum,
    MUINT*          puiFrameRateDen)
{
    if ((poMosMod == MNULL)
        || (puiFrameRateNum == MNULL)
        || (puiFrameRateDen == MNULL))
    {
        return LStatus_INVALID_PARAM;
    }

    *puiFrameRateNum = poMosMod->uiOutFrameRateNum;
    *puiFrameRateDen = poMosMod->uiOutFrameRateDen;

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       MosMod_SyncDeviceThread

Description:    .

\************************************************************************************************************/
LStatus MosMod_SyncDeviceThread(MosaicModule* poMosMod, MUINT uiSrcDtIdx, MUINT uiDstDtIdx)
{
    LStatus eStatus = LStatus_OK;

    if (uiSrcDtIdx != uiDstDtIdx)
    {
        eStatus = LDeviceThread_SyncDt(
                    poMosMod->ahDevThread[uiDstDtIdx],
                    poMosMod->ahDevThread[uiSrcDtIdx],
                    &poMosMod->oCpuThread.bKillThread);
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       MosMod_SyncDtOnAll

Description:    .

\************************************************************************************************************/
void MosMod_SyncDtOnAll(MosaicModule* poMosMod, MUINT uiDstDtIdx)
{
    MUINT i;

    for (i = 0; i < MOSAIC_VPC_COUNT; i++)
    {
        MosMod_SyncDeviceThread(poMosMod, MOSAIC_VPC_DT_IDX + i, uiDstDtIdx);
    }

    for (i = 0; i < MOSAIC_VPE_COUNT; i++)
    {
        MosMod_SyncDeviceThread(poMosMod, MOSAIC_VPE_DT_IDX + i, uiDstDtIdx);
    }
}

/************************************************************************************************************\

Function:       MosMod_SyncAllOnDt

Description:    .

\************************************************************************************************************/
void MosMod_SyncAllOnDt(MosaicModule* poMosMod, MUINT uiSrcDtIdx)
{
    MUINT i;

    for (i = 0; i < MOSAIC_VPC_COUNT; i++)
    {
        MosMod_SyncDeviceThread(poMosMod, uiSrcDtIdx, MOSAIC_VPC_DT_IDX + i);
    }

    for (i = 0; i < MOSAIC_VPE_COUNT; i++)
    {
        MosMod_SyncDeviceThread(poMosMod, uiSrcDtIdx, MOSAIC_VPE_DT_IDX + i);
    }
}

/************************************************************************************************************\

Function:       MosMod_ExecuteVpe

Description:    .

\************************************************************************************************************/
LStatus MosMod_ExecuteVpe(
    MosaicModule*           poMosMod,
    BufferInfo*             poSrcBuffer,
    BufferInfo*             poDstBuffer,
    MosaicInputParam*       poParam,
    LVP_Rect*               poDstRect,
    LClipRgn_Handle         hClipRgn,
    MBOOL                   bDoDtSync)
{
    const MUINT uiSliceCount = (poMosMod->bLowLatency ? sg_uiLowLatencySliceCount : 1) * MOSAIC_VPE_COUNT;

    MUINT i;

    if (bDoDtSync)
    {
        for (i = 1; i < MOSAIC_VPE_COUNT; i++)
        {
            MosMod_SyncDeviceThread(poMosMod, MOSAIC_VPE_DT_IDX, MOSAIC_VPE_DT_IDX + i);
        }
    }

    LVP_Rect oClipRect;
    LStatus  eStatus = LClipRgn_StartEnum(hClipRgn, LClipRgn_EnumDir_LEFT_DOWN);

    oClipRect.bEnable = MTRUE;

    while (LSTATUS_IS_SUCCESS(eStatus)
           && (LClipRgn_Enum(hClipRgn, &oClipRect.oRect) == LStatus_OK))
    {
        // Clip VPE in horizontal strips.
        MUINT uiTotalHeight  = oClipRect.oRect.iBottom - oClipRect.oRect.iTop;
        MUINT uiHeightPerVpe = AlignPow2((uiTotalHeight + uiSliceCount - 1) / uiSliceCount, 2);
        MUINT uiVpeIdx       = 0;

        while ((uiTotalHeight > 0) && LSTATUS_IS_SUCCESS(eStatus))
        {
            LVPE_Handle hVpe = poMosMod->ahVpe[uiVpeIdx];

            eStatus = LVPE_SetSrcRect(hVpe, &poParam->oSrcRect);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = LVPE_SetDstRect(hVpe, poDstRect);
            }

            if (uiHeightPerVpe > uiTotalHeight)
            {
                uiHeightPerVpe = uiTotalHeight;
            }

            oClipRect.oRect.iBottom = oClipRect.oRect.iTop + uiHeightPerVpe;

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = LVPE_SetClipRect(hVpe, &oClipRect);
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = LVPE_Execute(hVpe, poSrcBuffer->hBuffer, poDstBuffer->hBuffer);
            }

            oClipRect.oRect.iTop  = oClipRect.oRect.iBottom;
            uiTotalHeight        -= uiHeightPerVpe;

            uiVpeIdx++;

            if (uiVpeIdx >= MOSAIC_VPE_COUNT)
            {
                uiVpeIdx = 0;
            }
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       MosMod_ExecuteVpc

Description:    .

\************************************************************************************************************/
LStatus MosMod_ExecuteVpc(
    MosaicModule*           poMosMod,
    BufferInfo*             poSrcBuffer,
    BufferInfo*             poDstBuffer,
    MosaicInputParam*       poParam,
    LClipRgn_Handle         hClipRgn,
    MBOOL                   bDoDtSync)
{
    const MUINT uiSliceCount = (poMosMod->bLowLatency ? sg_uiLowLatencySliceCount : 1) * MOSAIC_VPC_COUNT;

    MUINT i;

    if (poParam->fAlpha != 1.0f)
    {
        for (i = 0; i < MOSAIC_VPE_COUNT; i++)
        {
            MosMod_SyncDeviceThread(poMosMod, MOSAIC_VPE_DT_IDX + i, MOSAIC_VPC_DT_IDX);
        }

        bDoDtSync = MTRUE;
    }

    if (bDoDtSync)
    {
        for (i = 1; i < MOSAIC_VPC_COUNT; i++)
        {
            MosMod_SyncDeviceThread(poMosMod, MOSAIC_VPC_DT_IDX, MOSAIC_VPC_DT_IDX + i);
        }
    }

    LVPC_SourceAlphaColorConstant oAlpha;
    LVP_Rect oClipRect;
    LVP_Rect oBlendRect;

    oClipRect.bEnable = MTRUE;

    if (poParam->fAlpha == 1.0f)
    {
        oBlendRect.bEnable = MFALSE;
    }
    else
    {
        oBlendRect = poParam->oDstRect;
    }

    oAlpha.bYCbCrA      = MFALSE;
    oAlpha.oColor.fYR   = 1.0f;
    oAlpha.oColor.fCbG  = 1.0f;
    oAlpha.oColor.fCrB  = 1.0f;

    LStatus eStatus = LClipRgn_StartEnum(hClipRgn, LClipRgn_EnumDir_LEFT_DOWN);

    while (LSTATUS_IS_SUCCESS(eStatus)
           && (LClipRgn_Enum(hClipRgn, &oClipRect.oRect) == LStatus_OK))
    {
        MUINT uiTotalLines  = oClipRect.oRect.iBottom - oClipRect.oRect.iTop;
        MUINT uiLinesPerVpc = AlignPow2((uiTotalLines + uiSliceCount - 1) / uiSliceCount, 4);

        for (i = 0; (i < uiSliceCount) && (uiTotalLines > 0) && LSTATUS_IS_SUCCESS(eStatus); i++)
        {
            LVPC_Handle hVpc = poMosMod->ahVpc[i % MOSAIC_VPC_COUNT];

            eStatus = LVPC_SetSrcRect(hVpc, 0, &poParam->oSrcRect);

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = LVPC_SetDstRect(hVpc, 0, &poParam->oDstRect);
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                oAlpha.oColor.fAlpha = poParam->fAlpha;

                eStatus = LVPC_SetSourceParam(
                                hVpc,
                                0,
                                LVPC_Source_ALPHA_COLOR_CONSTANT,
                                sizeof(oAlpha),
                                &oAlpha);

            }

            if (oBlendRect.bEnable)
            {
                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    eStatus = LVPC_SetSrcRect(hVpc, 1, &oBlendRect);
                }

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    eStatus = LVPC_SetDstRect(hVpc, 1, &poParam->oDstRect);
                }

                if (LSTATUS_IS_SUCCESS(eStatus))
                {
                    oAlpha.oColor.fAlpha = 1.0f - poParam->fAlpha;

                    eStatus = LVPC_SetSourceParam(
                                    hVpc,
                                    1,
                                    LVPC_Source_ALPHA_COLOR_CONSTANT,
                                    sizeof(oAlpha),
                                    &oAlpha);
                }
            }

            if (uiLinesPerVpc > uiTotalLines)
            {
                uiLinesPerVpc = uiTotalLines;
            }

            oClipRect.oRect.iBottom = oClipRect.oRect.iTop + uiLinesPerVpc;

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = LVPC_SetClipRect(hVpc, &oClipRect);
            }

            if (LSTATUS_IS_SUCCESS(eStatus))
            {
                eStatus = LVPC_Execute(hVpc, poSrcBuffer->hBuffer, poDstBuffer->hBuffer, oBlendRect.bEnable);
            }

            oClipRect.oRect.iTop  = oClipRect.oRect.iBottom;
            uiTotalLines         -= uiLinesPerVpc;
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       MosMod_CpuThread

Description:    .

\************************************************************************************************************/
LStatus MosMod_CpuThread(void* pvData)
{
    if (pvData == MNULL)
    {
        MsgLogErr("ERROR! NULL data.");
        return LStatus_INVALID_PARAM;
    }

    MosaicModule* poMosMod = (MosaicModule*)pvData;

    MUINT   uiEosCount              = 0;
    MUINT64 uiPresentTimeNsec       = 0;
    MUINT64 uiOutFrameDurationNsec  = ((MUINT64)poMosMod->uiOutFrameRateDen * 1000 * 1000 * 1000)
                                      / poMosMod->uiOutFrameRateNum;
    MINT64  iMaxTimeoutNsec         = GetMonoTimeNsec();
    MBOOL   bWarmingUp              = MTRUE;
    MUINT64 uiLastMaxTimestampUsec  = 0;
    MINT64  uiLastMaxTimeOffsetNsec = 0;
    MUINT   i;

    ModThread_SetName(poMosMod->szModuleName);
    MsgLog(2, "Start thread %p.", pthread_self());

    while (!poMosMod->oCpuThread.bKillThread)
    {
        BufferInfo* poDstBuffer;
        LStatus     eStatus = ModLnk_GetReturnedBuffer(
                                        &(poMosMod->oOutLink),
                                        100,
                                        poMosMod->ahDevThread[MOSAIC_VPE_DT_IDX],
                                        &poDstBuffer);

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            MBOOL bWarmingDone          = bWarmingUp;
            MUINT uiFullScreenInputIdx  = ~0;

            if (poMosMod->bShowFullScreen)
            {
                uiFullScreenInputIdx = poMosMod->uiFullScreenInputIdx;
            }
            else if (poDstBuffer->pvPrivateData != MNULL)
            {
                MBOOL* pbValidBuffer = (MBOOL*)(poDstBuffer->pvPrivateData);

                if (!(*pbValidBuffer))
                {
                    LColor oColor;

                    MsgLog(4, "Clear buffer[%u]", poDstBuffer->uiId);

                    oColor.ePixelFormat = LPixelFormat_B8G8R8A8;
                    oColor.uiAlpha      = 0;
                    oColor.uiRed        = 0;
                    oColor.uiGreen      = 0;
                    oColor.uiBlue       = 0;

                    LBlit_SolidFill(
                        poMosMod->hBlit,
                        poDstBuffer->hBuffer,
                        &oColor,
                        MNULL,
                        MNULL,
                        LBlit_Bop_S,
                        MNULL);

                    *pbValidBuffer = MTRUE;
                }
            }

            MosMod_SyncAllOnDt(poMosMod, MOSAIC_VPE_DT_IDX);

            MUINT64 uiMaxTimestampUsec  = 0;
            MINT64  uiMaxTimeOffsetNsec = 0;

            for (i = 0; i < poMosMod->uiInputCount; i++)
            {
                MosaicInputData*  poInData = poMosMod->aoInData + i;
                ModuleLinkInput*  poInLink = poMosMod->aoInLink + poInData->uiInputIndex;
                MosaicInputParam* poParam  = poMosMod->aoInParam + poInData->uiInputIndex;

                MBOOL bUseVpc   = (poParam->bUseVpc && (uiFullScreenInputIdx != poInData->uiInputIndex))
                                  ? MTRUE : MFALSE;
                MBOOL bDoDtSync = MFALSE;

                while ((poInData->uiPresentTimeNsec <= uiPresentTimeNsec)
                       && !poInData->bEos
                       && !poMosMod->oCpuThread.bKillThread)
                {
                    BufferInfo* poSrcBuffer;
                    MUINT       uiQueueLen;
                    MUINT64     uiCurTimeNsec = GetMonoTimeNsec();
                    MUINT64     uiTimeoutNsec = iMaxTimeoutNsec - min(iMaxTimeoutNsec, uiCurTimeNsec);

                    while (MTRUE)
                    {
                        eStatus = ModLnkIn_GetSubmittedBuffer(
                                        poInLink,
                                        uiTimeoutNsec / (1000 * 1000),
                                        0,
                                        bUseVpc ? poMosMod->ahDevThread[MOSAIC_VPC_DT_IDX]
                                                : poMosMod->ahDevThread[MOSAIC_VPE_DT_IDX],
                                        &poSrcBuffer,
                                        &bDoDtSync,
                                        &uiQueueLen);

                        if (poMosMod->bSyncInputs && !bWarmingUp && LSTATUS_IS_SUCCESS(eStatus))
                        {
                            uiCurTimeNsec = GetMonoTimeNsec();
                            uiTimeoutNsec = iMaxTimeoutNsec - min(iMaxTimeoutNsec, uiCurTimeNsec);

                            if ((uiTimeoutNsec > 0) || (uiQueueLen > 0))
                            {
                                MUINT64 uiMaxTimeUsec = uiLastMaxTimestampUsec
                                                        + (poInData->uiFrameDurationNsec / 2000);

                                if ((poSrcBuffer->uiTimestampUsec < uiMaxTimeUsec)
                                    && !poMosMod->oCpuThread.bKillThread)
                                {
                                    ModLnkIn_ReturnBuffer(poInLink, poSrcBuffer, MNULL, NO_TAG);

                                    MsgLog(2, "InLink[%u]: Skip frame: Timestamp= %lu usec, Max= %lu usec",
                                              poInData->uiInputIndex, poSrcBuffer->uiTimestampUsec,
                                              uiMaxTimeUsec);
                                    continue;
                                }
                            }

                            poInData->iTimestampOffsetNsec = uiLastMaxTimeOffsetNsec;

                            MsgLog(4, "InLink[%u]: Timestamp= %lu msec",
                                      i, poSrcBuffer->uiTimestampUsec / 1000);
                        }

                        break;
                    }

                    if (LSTATUS_IS_SUCCESS(eStatus))
                    {
                        if (poInData->poCurBuffer != MNULL)
                        {
                            MosMod_SyncDtOnAll(poMosMod, MOSAIC_SYNC_DT_IDX);

                            ModLnkIn_ReturnBuffer(
                                        poInLink,
                                        poInData->poCurBuffer,
                                        poMosMod->ahDevThread[MOSAIC_SYNC_DT_IDX],
                                        NO_TAG);
                            poInData->poCurBuffer = MNULL;
                        }

                        if (poSrcBuffer->bEndOfStream)
                        {
                            ModLnkIn_ReturnBuffer(poInLink, poSrcBuffer, MNULL, NO_TAG);
                            MosMod_InvalidateBuffers(poMosMod);

                            poInData->bEos = MTRUE;
                            uiEosCount++;

                            MsgLog(2, "InLink[%u]: END-OF-STREAM (%u/%u inputs are EOS).",
                                      poInData->uiInputIndex, uiEosCount, poMosMod->uiInputCount);
                            break;
                        }

                        poInData->poCurBuffer = poSrcBuffer;

                        MUINT64 uiTimestampNsec = (poSrcBuffer->uiTimestampUsec * 1000)
                                                  + poInData->iTimestampOffsetNsec;
                        MINT64  iTimeDriftNsec  = (MINT64)uiTimestampNsec - poInData->uiPresentTimeNsec;

                        if (labs(iTimeDriftNsec) > poInData->uiMaxTimeDriftNsec)
                        {
                            if (labs(iTimeDriftNsec) < poInData->uiFrameDurationNsec)
                            {
                                // Timestamp is invalid if timestamp offset = 0
                                if (poInData->iTimestampOffsetNsec != 0)
                                {
                                    // Small drifting is probably due to an accumulated error in present time
                                    // calculation. So, sync present time on the timestamp.
                                    poInData->uiPresentTimeNsec = uiTimestampNsec;
                                }

                                MsgLog(2, "InLink[%u]: Small drifting (%ld nsec)",
                                          poInData->uiInputIndex, iTimeDriftNsec);
                            }
                            else
                            {
                                // Sudden large drifting probably means that the timestamp is wrong.
                                // So, sync timestamp on the present time.
                                poInData->iTimestampOffsetNsec -= iTimeDriftNsec;

                                MsgLog(2, "InLink[%u]: Large drifting (%ld nsec)",
                                          poInData->uiInputIndex, iTimeDriftNsec);
                            }
                        }

                        if (uiQueueLen <= poInData->uiMaxQueueLenAllowed)
                        {
                            poInData->uiPresentTimeNsec += poInData->uiFrameDurationNsec;
                        }
                        else
                        {
                            // Queue overflow: Skip one frame
                            poInData->iTimestampOffsetNsec -= poInData->uiFrameDurationNsec;

                            MsgLog(2, "InLink[%u]: Queue overflow (Q-Len= %u).",
                                      poInData->uiInputIndex, uiQueueLen);
                        }

                        if (poSrcBuffer->uiTimestampUsec > uiMaxTimestampUsec)
                        {
                            uiMaxTimestampUsec  = poSrcBuffer->uiTimestampUsec;
                            uiMaxTimeOffsetNsec = poInData->iTimestampOffsetNsec;
                        }
                    }
                    else
                    {
                        if (eStatus == LStatus_TIMEOUT)
                        {
                            // Queue underflow: Pause one frame
                            poInData->uiPresentTimeNsec    += poInData->uiFrameDurationNsec;
                            poInData->iTimestampOffsetNsec += poInData->uiFrameDurationNsec;

                            MsgLog(2, "InLink[%u]: Queue underflow.", poInData->uiInputIndex);
                        }

                        break;
                    }
                }

                if (uiEosCount == poMosMod->uiInputCount)
                {
                    poDstBuffer->bEndOfStream = MTRUE;
                    poMosMod->oCpuThread.bKillThread = MTRUE;
                    break;
                }

                if (poInData->poCurBuffer != MNULL)
                {
                    if (uiFullScreenInputIdx == ~0)
                    {
                        if (poParam->bUseVpc)
                        {
                            MosMod_ExecuteVpc(
                                poMosMod,
                                poInData->poCurBuffer,
                                poDstBuffer,
                                poParam,
                                poInData->hClipRgn,
                                bDoDtSync);
                        }
                        else
                        {
                            MosMod_ExecuteVpe(
                                poMosMod,
                                poInData->poCurBuffer,
                                poDstBuffer,
                                poParam,
                                &poParam->oDstRect,
                                poInData->hClipRgn,
                                bDoDtSync);
                        }
                    }
                    else if (uiFullScreenInputIdx == poInData->uiInputIndex)
                    {
                        MosMod_ExecuteVpe(
                            poMosMod,
                            poInData->poCurBuffer,
                            poDstBuffer,
                            poParam,
                            &poMosMod->oFullDstRect,
                            poMosMod->hFullClipRgn,
                            bDoDtSync);

                        MosMod_InvalidateBuffers(poMosMod);
                    }

                    poDstBuffer->uiSyncPtsUsec = poInData->poCurBuffer->uiSyncPtsUsec;
                }
                else
                {
                    bWarmingDone = MFALSE;
                }
            }

            uiLastMaxTimestampUsec  = uiMaxTimestampUsec;
            uiLastMaxTimeOffsetNsec = uiMaxTimeOffsetNsec;

            if (sg_bEnableLatencyProfile)
            {
                poDstBuffer->uiTimestampUsec = uiMaxTimestampUsec;
            }
            else
            {
                poDstBuffer->uiTimestampUsec = uiPresentTimeNsec / 1000;
            }

            MosMod_SyncDtOnAll(poMosMod, MOSAIC_SYNC_DT_IDX);

            // Don't submit buffers until we received buffers at input.
            if(!bWarmingUp || bWarmingDone)
            {
                ModLnk_SubmitBuffer(
                            &(poMosMod->oOutLink),
                            poDstBuffer,
                            poMosMod->ahDevThread[MOSAIC_SYNC_DT_IDX],
                            NO_TAG);
            }
            else
            {
                ModLnk_ReleaseBuffer(
                            &(poMosMod->oOutLink),
                            poDstBuffer);
            }

            if (poMosMod->pfnSubmitNotify != MNULL)
            {
                ((MosMod_SubmitNotifyFn)poMosMod->pfnSubmitNotify)(poMosMod, uiEosCount > 0);
            }

            uiPresentTimeNsec += uiOutFrameDurationNsec;
            iMaxTimeoutNsec   += uiOutFrameDurationNsec;

            if (bWarmingDone)
            {
                // Now that the warming up is completed, give more time to get submitted buffers.
                bWarmingUp = MFALSE;
                iMaxTimeoutNsec += (max(poMosMod->oOutLink.uiBufferCount, 2) - 2) * uiOutFrameDurationNsec;

                MsgLog(2, "Warming up done.");
            }
        }

        // Patch: return buffers to the decoder connected at input if it is trying to stop.
        for (i = 0; i < poMosMod->uiInputCount; i++)
        {
            MosaicInputData*  poInData = poMosMod->aoInData + i;
            ModuleLinkInput*  poInLink = poMosMod->aoInLink + poInData->uiInputIndex;

            poInLink->bSkipAll = poInLink->poModLnk->bSkipAll;

            if (poInLink->bSkipAll)
            {
                if (poInData->poCurBuffer != MNULL)
                {
                    MosaicInputParam* poParam = poMosMod->aoInParam + poInData->uiInputIndex;

                    ModLnkIn_ReturnBuffer(
                        poInLink,
                        poInData->poCurBuffer,
                        poMosMod->ahDevThread[poParam->bUseVpc ? MOSAIC_VPC_DT_IDX : MOSAIC_VPE_DT_IDX],
                        NO_TAG);

                    poInData->poCurBuffer = MNULL;
                }

                ModLnkIn_CancelSubmittedBuffers(poInLink);
            }
        }

        if (LSTATUS_IS_FAIL(eStatus)
            && (eStatus != LStatus_TIMEOUT))
        {
            usleep(1000);
        }
    }

    for (i = 0; i < poMosMod->uiInputCount; i++)
    {
        MosaicInputData* poInData = poMosMod->aoInData + i;

        if (poInData->poCurBuffer != MNULL)
        {
            MosaicInputParam* poParam = poMosMod->aoInParam + poInData->uiInputIndex;

            ModLnkIn_ReturnBuffer(
                poMosMod->aoInLink + poInData->uiInputIndex,
                poInData->poCurBuffer,
                poMosMod->ahDevThread[poParam->bUseVpc ? MOSAIC_VPC_DT_IDX : MOSAIC_VPE_DT_IDX],
                NO_TAG);
        }
    }

    MsgLog(2, "Kill thread.");

    return LStatus_OK;
}

/************************************************************************************************************\

Function:       MosMod_BuildAllClipRgn

Description:    .

\************************************************************************************************************/
LStatus MosMod_BuildAllClipRgn(MosaicModule* poMosMod)
{
    LStatus eStatus = LStatus_OK;

    MUINT i;
    for (i = 0; (i < poMosMod->uiInputCount) && LSTATUS_IS_SUCCESS(eStatus); i++)
    {
        MosaicInputData*  poInData = poMosMod->aoInData + i;
        MosaicInputParam* poParam  = poMosMod->aoInParam + poInData->uiInputIndex;

        if (poInData->hClipRgn == MNULL)
        {
            eStatus = LClipRgn_Create(&poInData->hClipRgn);
        }

        if (LSTATUS_IS_SUCCESS(eStatus))
        {
            eStatus = LClipRgn_SetRect(poInData->hClipRgn, &poParam->oDstRect.oRect, LBlit_Bop_S);
        }

        if (poParam->fAlpha == 1.0f)
        {
            // Remove overlap drawing: Clip out the input regions masked by this input region.
            MUINT j;
            for (j = 0; (j < i) && LSTATUS_IS_SUCCESS(eStatus); j++)
            {
                eStatus = LClipRgn_SetClipRgn(
                                poMosMod->aoInData[j].hClipRgn,
                                poInData->hClipRgn,
                                LBlit_Bop_DSna);
            }
        }
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       MosMod_BuildAllClipRgn

Description:    .

\************************************************************************************************************/
LStatus MosMod_MoveDstRect(
            MosaicModule*   poMosMod,
            MUINT           uiInputIndex,
            const LPOS*     poOffset)
{
    MosaicInputParam* poParam = poMosMod->aoInParam + uiInputIndex;

    MUINT i;
    for (i = 0; i < poMosMod->uiInputCount; i++)
    {
        if (poMosMod->aoInData[i].uiInputIndex == uiInputIndex)
        {
            poParam->oDstRect.oRect.iLeft   += poOffset->iX;
            poParam->oDstRect.oRect.iTop    += poOffset->iY;
            poParam->oDstRect.oRect.iRight  += poOffset->iX;
            poParam->oDstRect.oRect.iBottom += poOffset->iY;

            return MosMod_BuildAllClipRgn(poMosMod);
        }
    }

    return LStatus_FAIL;
}

/************************************************************************************************************\

Function:       MosMod_Start

Description:    .

\************************************************************************************************************/
LStatus MosMod_Start(MosaicModule* poMosMod)
{
    MsgLog(2, "{...");

    if (poMosMod == MNULL)
    {
        MsgLogErr("ERROR! Invalid parameter.");
        return LStatus_INVALID_PARAM;
    }

    MBOOL bVpeCreated     = MTRUE;
    MBOOL bVpcCreated     = MTRUE;
    MBOOL bInputConnected = MFALSE;
    MUINT i;

    for (i = 0; i < MOSAIC_VPE_COUNT; i++)
    {
        if (poMosMod->ahVpe[i] == MNULL)
        {
            bVpeCreated = MFALSE;
        }
    }

    for (i = 0; i < MOSAIC_VPC_COUNT; i++)
    {
        if (poMosMod->ahVpc[i] == MNULL)
        {
            bVpcCreated = MFALSE;
        }
    }

    for (i = 0; i < MOSAIC_MAX_INPUT; i++)
    {
        if (poMosMod->aoInLink[i].poModLnk != MNULL)
        {
            bInputConnected = MTRUE;
            break;
        }
    }

    if (!bVpeCreated
        || !bVpcCreated
        || (poMosMod->hBlit == MNULL)
        || !bInputConnected
        || (poMosMod->oOutLink.uiSubmitCount == 0))
    {
        MsgLogErr("ERROR! No VPE handle or bad connection.");
        return LStatus_FAIL;
    }

    // Init input dynamic data.

    poMosMod->uiInputCount = 0;

    MUINT   uiZOrder        = 0;
    MUINT   uiZOrderNext    = ~0;
    MBOOL   bGetNext        = MTRUE;
    LStatus eStatus         = LStatus_OK;

    while (bGetNext && LSTATUS_IS_SUCCESS(eStatus))
    {
        bGetNext = MFALSE;

        for (i = 0; (i < MOSAIC_MAX_INPUT) && LSTATUS_IS_SUCCESS(eStatus); i++)
        {
            if (poMosMod->aoInLink[i].poModLnk != MNULL)
            {
                MosaicInputParam* poParam = poMosMod->aoInParam + i;

                // Sort inputs in z order incrementally.
                if (poParam->uiZOrder == uiZOrder)
                {
                    MosaicInputData* poInData = poMosMod->aoInData + poMosMod->uiInputCount;

                    poInData->uiInputIndex          = i;
                    poInData->iTimestampOffsetNsec  = 0;
                    poInData->uiFrameDurationNsec   = ((MUINT64)poParam->uiFrameRateDen * 1000 * 1000 * 1000)
                                                      / poParam->uiFrameRateNum;
                    poInData->uiPresentTimeNsec     = 0;
                    poInData->uiMaxTimeDriftNsec    = poInData->uiFrameDurationNsec / 4;
                    poInData->uiMaxQueueLenAllowed  = poMosMod->bLowLatency
                                                      ? 1
                                                      : poParam->bControlQueueOverflow
                                                        ? (poMosMod->aoInLink[i].poModLnk->uiBufferCount - 4)
                                                        : (poMosMod->aoInLink[i].poModLnk->uiBufferCount * 2);
                    poInData->poCurBuffer           = MNULL;
                    poInData->bEos                  = MFALSE;

                    MsgLog(2, "InputData[%u]: In-Idx= %u, Z= %u, Frame duration= %lu nsec.",
                              poMosMod->uiInputCount, i, poParam->uiZOrder, poInData->uiFrameDurationNsec);

                    poMosMod->uiInputCount++;
                }
                else if ((poParam->uiZOrder > uiZOrder)
                         && (poParam->uiZOrder < uiZOrderNext))
                {
                    uiZOrderNext = poParam->uiZOrder;
                    bGetNext     = MTRUE;
                }
            }
        }

        uiZOrder     = uiZOrderNext;
        uiZOrderNext = ~0;
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        eStatus = MosMod_BuildAllClipRgn(poMosMod);
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        if (poMosMod->uiInputCount > 0)
        {
            eStatus = ModThread_Start(&(poMosMod->oCpuThread), poMosMod, MosMod_CpuThread);
        }
        else
        {
            eStatus = LStatus_FAIL;
        }
    }

    if (LSTATUS_IS_SUCCESS(eStatus))
    {
        MsgLog(2, "...} (status= %d - %s)", eStatus, GetStatusStr(eStatus));
    }
    else
    {
        MsgLogErr("ERROR! status= %d (%s)", eStatus, GetStatusStr(eStatus));
    }

    return eStatus;
}

/************************************************************************************************************\

Function:       MosMod_Stop

Description:    .

\************************************************************************************************************/
void MosMod_Stop(MosaicModule* poMosMod)
{
    MsgLog(2, "{...");

    if (poMosMod != MNULL)
    {
        ModThread_Stop(&(poMosMod->oCpuThread));

        MUINT i;
        for (i = 0; i < MOSAIC_MAX_INPUT; i++)
        {
            poMosMod->aoInLink[i].bSkipAll = MFALSE;
        }
    }

    MsgLog(2, "...}");
}

/************************************************************************************************************\

Function:       MosMod_ShowOneInput

Description:    .

\************************************************************************************************************/
void MosMod_ShowOneInputFullScreen(MosaicModule* poMosMod, MBOOL bEnable, MUINT uiInputIndex)
{
    if (poMosMod != MNULL)
    {
        if (bEnable)
        {
            if (uiInputIndex < poMosMod->uiInputCount)
            {
                poMosMod->uiFullScreenInputIdx = uiInputIndex;
            }
        }

        poMosMod->bShowFullScreen = bEnable;
    }
}
